#ifndef __AUDIT_H
#define __AUDIT_H

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include <sys/audit.h>

/* memory allocation - this header is used to identify memory allocated
 * by this library */

#define __AUD_FLAVOR_STATIC      0x0000      /* memory was not malloc()'d */
#define __AUD_FLAVOR_DYNAMIC     0x8000      /* memory was malloc()'d */

#define __AUD_MAGIC_EVINFO       1      /* aud_evinfo_t */
#define __AUD_MAGIC_HDR          2      /* aud_hdr_t */
#define __AUD_MAGIC_OBJ          3      /* aud_obj_t */
#define __AUD_MAGIC_SUBJ         4      /* aud_subj_t */
#define __AUD_MAGIC_INFO         5      /* aud_info_t */
#define __AUD_MAGIC_RECORD       6      /* aud_rec_t */
#define __AUD_MAGIC_PURPOSE_     7      /* internal use for keeping tabs on
					   memory allocated by library */
#define __AUD_MAGIC_DATA         8      /* used for unstructured data */

/* internal structures */

struct __aud_info_s {
    int magic;                                   /* __AUD_MAGIC_INFO */
    int item_id; /* lookup/indexing id (see item_id in <sys/audit.h> */
    aud_info_t datum;                            /* actual data item */
    struct __aud_info_s *next;                   /* linked list */
};

/* structures opaque in the <sys/audit.h> file are mapped to these */

struct __aud_section_s {
    int magic;                                   /* __AUD_MAGIC_* */
    int count;                                   /* total # of data items */
    int *owner_count;                            /* ptr to # of events */
    int prev_item_id;                            /* previous call's id */
    struct __aud_section_s **before, *after;     /* linking list */
    struct __aud_info_s *data;
};

struct __aud_rec_s {
    int magic;
    int event_count;
    struct __aud_section_s *event;
    int header_count;
    struct __aud_section_s *header;
    int object_count;
    struct __aud_section_s *object;
    int subject_count;
    struct __aud_section_s *subject;
};

/* XXX - this should align on a 64-bit boundary on all systems
 * it is used for keeping track of memory allocated by libaudit */

struct purpose {
    int magic;         /* magic to indicate that this is a purpose block */
    void *data;        /* pointer to the actual data */
    int length;        /* length of actual data */
    int flavor;        /* flavor of actual data (top bit is high for
			* dynamic memory) */
};

/* aud_alloc.c */

extern void *__aud_malloc(int size, int type);
extern void *__aud_lookup(const void *old);
extern int __aud_free(const void *old);

/* aud_generic.c - aud_delete/get/put_*_*() fns are mapped to these. */

extern int __aud_delete_section(int magic, const void *section_d);
extern int __aud_delete_section_info(int magic, const void *section_d,
				     int item_id);
extern int __aud_get_section(int magic, aud_rec_t ar, int index,
			     const void **section_p);
extern int __aud_get_section_info(int magic, const void *section_d,
				  int item_id,
				  const aud_info_t **aud_sect_info_p);
extern int __aud_put_section(int magic, aud_rec_t ar,
			     const struct __aud_section_s **next_p,
			     const struct __aud_section_s **new_p);
extern int __aud_put_section_info(int magic, const void *section_d,
				  int position, int item_id,
				  const aud_info_t *aud_event_info_p);

#endif
