/*
 * inotify-lib.c - backend for really simple inotify example
 *
 * Robert Love <rml@novell.com>
 */

#include <config.h>

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <glib.h>

#include "inotify-lib.h"

/*
 * inotify_open - open the inotify device and return a fresh GIOChannel
 * or NULL on error.
 */
GIOChannel *
inotify_open (void)
{
	GIOChannel *gio;
	GError *err = NULL;

	gio = g_io_channel_new_file ("/dev/inotify", "r", &err);
	if (!gio) {
		g_warning ("Failed to open /dev/inotify: %s\n", err->message);
		g_error_free (err);
	}

	return gio;
}

/*
 * inotify_close - close the GIOChannel
 */
void
inotify_close (GIOChannel *gio)
{
	g_io_channel_close (gio);
}

/*
 * inotify_add_watch - Add an inotify watch on the object "name" to the
 * open inotify instance associated with "gio".  The user may do this any
 * number of times, even on the same device instance.
 */
int
inotify_add_watch (GIOChannel *gio, const char *name)
{
	struct inotify_watch_request request;
	int dev_fd, file_fd, ret;

	dev_fd = g_io_channel_unix_get_fd (gio);
	file_fd = open (name, O_RDONLY);
	if (file_fd < 0) {
		perror ("open");
		return -1;
	}

	request.fd = file_fd;
	request.mask = IN_ALL_EVENTS;

	ret = ioctl (dev_fd, INOTIFY_WATCH, &request);
	if (ret < 0)
		perror ("ioctl");

	if (close (file_fd))
		perror ("close");

	return ret;
}

/* inotify lets us slurp a lot of events at once.  we go with a nice big 32k */
#define INOTIFY_BUF	32768

/*
 * __inotify_handle_cb - our internal GIOChannel G_IO_IN callback.  Slurps as
 * many events as are available and calls the user's callback (given as "data")
 * for each event.  If any invocations of the user's callback return FALSE, so
 * do we, terminating this watch.  Otherwise, we return TRUE.
 */
static gboolean
__inotify_handle_cb (GIOChannel *gio, GIOCondition condition, gpointer data)
{
	char buf[INOTIFY_BUF];
	InotifyCb f = data;
	GIOError err;
	guint len;
	int i = 0;

	/* read in as many pending events as we can */
	err = g_io_channel_read (gio, buf, INOTIFY_BUF, &len);
	if (err != G_IO_ERROR_NONE) {
		g_warning ("Error reading /dev/inotify: %d\n", err);
		return FALSE;
	}

	/* reconstruct each event and send to the user's callback */
	while (i < len) {
		const char *name = "The watch";
		struct inotify_event *event;

		event = (struct inotify_event *) &buf[i];
		if (event->len)
			name = &buf[i] + sizeof (struct inotify_event);

		if (f (name, event->wd, event->mask, event->cookie) == FALSE)
			return FALSE;

		i += sizeof (struct inotify_event) + event->len;
	}

	return TRUE;
}

/*
 * inotify_callback - associate a user InotifyCb callback with the given
 * GIOChannel.  This is normally done but once.
 */
void
inotify_callback (GIOChannel *gio, InotifyCb f)
{
	g_io_add_watch (gio, G_IO_IN, __inotify_handle_cb, f);
}
